<?php

namespace App\Http\Controllers;

use Barryvdh\DomPDF\Facade\Pdf;
use DateTime;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\ValidationException;

define('SET_ROWNUM', 'set @rownum=0');
define('ROWNUM_EXPRESSION', '@rownum := @rownum + 1 as rownum');
class Admin extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function creacionCursos()
    {
        $view_creacion_cursos = 'admin/creacion_cursos';
        $secciones = [];
        $clases = [];
        if (session()->get('redirigido_step2', false)) {
            session()->forget('redirigido_step2');
            $secciones = $this->obtenerSecciones(session()->get('id_curso'));
        } elseif (session()->get('redirigido_step3', false)) {
            session()->forget('redirigido_step3');
            $clases = $this->obtenerClases(session()->get('id_seccion'));
        } else {
            session()->forget('id_curso');
            session()->forget('id_seccion');
            session()->put('tab', 'step1');
        }
        return view($view_creacion_cursos, compact('secciones', 'clases'));
    }

    public function obtenerCursos($paginacion)
    {
        return DB::table('Curso')
            ->join('Instructor', 'Curso.Instructor_idInstructor', '=', 'Instructor.idInstructor')
            ->join('users', 'Instructor.users_id', '=', 'users.id')
            ->select(
                'Curso.Nombre as nombre',
                DB::raw("CONCAT(users.name, ' ', users.last) as instructor"),
                DB::raw("CONCAT(Curso.Precio, ' / ', IFNULL(Curso.Precio_sin_descuento, Curso.Precio)) as precio"),
                'Curso.idCurso as id'
            )
            ->whereIn('Curso.Estado', [0, 1])
            ->orderByRaw('CAST(posicion AS UNSIGNED) ASC')
            ->paginate($paginacion);
    }

    public function obtenerCurso($id_curso)
    {
        $curso = DB::table('Curso')
            ->select(
                'Curso.Nombre as nombre',
                'Curso.Precio_sin_descuento as precio_original',
                'Curso.Precio as precio_actual',
                'Curso.Modulo as cantidad_cursos',
                'Curso.CantidadPaquetes as cantidad_paquetes',
                'Curso.Descripcion as descripcion',
                DB::raw('IF(Curso.numerocurso != 0, 0, 1) as corona'),
                'Curso.Contenido as contenido'
            )
            ->where('Curso.idCurso', $id_curso)
            ->first();

        return response()->json(['curso' => $curso]);
    }

    public function registrarCurso(Request $request)
    {
        try {
            DB::beginTransaction();

            $messages = [
                'nombre_curso.required' => 'El nombre del curso es requerido.',
                'precio_actual.required' => 'El precio actual es requerido.',
                'precio_actual.numeric' => 'El precio actual debe ser un valor numérico.',
                'imagen_curso.required' => 'La imagen del curso es requerida.',
                'imagen_curso.image' => 'La imagen debe ser un archivo de imagen válido.',
                'imagen_curso.mimes' => 'La imagen debe ser en formato JPEG, PNG, JPG, SVG o WEBP.',
                'imagen_curso.max' => 'La imagen excede el tamaño permitido.',
                'cantidad_cursos.integer' => 'El número de cursos debe ser un valor entero.',
                'cantidad_paquetes.integer' => 'El número de paquetes debe ser un valor entero.',
                'descripcion.required' => 'La descripción del curso es requerida.',
                'contenido.required' => 'El contenido del curso es requerido.',
            ];
            $request->validate([
                'nombre_curso' => 'required',
                'precio_actual' => 'required|numeric',
                'imagen_curso' => 'required|image|mimes:jpeg,png,jpg,svg,webp|max:4000',
                'cantidad_cursos' => 'nullable|integer',
                'cantidad_paquetes' => 'nullable|integer',
                'descripcion' => 'required',
                // 'corona' => 'required|boolean', # Validar input 'corona' (boolean)
                'contenido' => 'required',
            ], $messages);

            $nombre = $request->input('nombre_curso');
            $precio_actual = $request->input('precio_actual');
            $precio_original = $request->input('precio_original');
            $filename = uniqid() . '_' . time() . '.' . $request->imagen_curso->getClientOriginalExtension();
            $request->imagen_curso->move(public_path('images'), $filename);

            $modulo = $request->input('cantidad_cursos');
            $packs = $request->input('cantidad_paquetes');
            $descripcion = $request->input('descripcion');
            $contenido = $request->input('contenido');
            $corona = $request->filled('corona'); #boolean

            # Obtener el maximo de la posicion
            $maxPosition = DB::table('Curso')->max(DB::raw('CAST(posicion AS UNSIGNED)')); # (CAST: Convertir a entero)

            $data = array(
                'Nombre' => $nombre,
                "Precio" => $precio_actual,
                "Precio_sin_descuento" => $precio_original,
                "Imagen" => "images/" . $filename,
                "Modulo" => $modulo, # Cantidad de cursos
                "CantidadPaquetes" => $packs, # Cantida de paquetes
                "Contenido" => $contenido,
                "Descripcion" => $descripcion,
                "Instructor_idInstructor" => 1,
                "Categoria_idCategoria" => 1,
                "posicion" => $maxPosition + 1,
                "numerocurso" => !$corona ? 1 : 0,
                "Estado" => 1,
                "tipoCurso" => null,
            );

            $id_curso = DB::table('Curso')->insertGetId($data);

            # avanzar al siguiente step
            session()->put('tab', 'step2');
            session()->put('id_curso', $id_curso);
            session()->put('redirigido_step2', true);

            DB::commit();
            return redirect()->back();
        } catch (ValidationException $e) {
            DB::rollback();
            // Manejo del error de validación
            return back()->withInput()->withErrors($e->errors());
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al crear el curso: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al crear el curso. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function actualizarCurso(Request $request)
    {
        try {
            DB::beginTransaction();

            $messages = [
                'id_curso.required' => 'Error al registrar la seccion. Por favor, inténtalo de nuevo. (001)',
                'id_curso.integer' => 'Error al registrar la seccion. Por favor, inténtalo de nuevo. (002)',
                'nombre_curso.required' => 'El nombre del curso es requerido.',
                'precio_actual.required' => 'El precio actual es requerido.',
                'precio_actual.numeric' => 'El precio actual debe ser un valor numérico.',
                'imagen_curso.image' => 'La imagen debe ser un archivo de imagen válido.',
                'imagen_curso.mimes' => 'La imagen debe ser en formato JPEG, PNG, JPG, SVG o WEBP.',
                'imagen_curso.max' => 'La imagen excede el tamaño permitido.',
                'cantidad_cursos.integer' => 'El número de cursos debe ser un valor entero.',
                'cantidad_paquetes.integer' => 'El número de paquetes debe ser un valor entero.',
                'descripcion.required' => 'La descripción del curso es requerida.',
                'contenido.required' => 'El contenido del curso es requerido.',
            ];
            $request->validate([
                'id_curso' => 'required|integer',
                'nombre_curso' => 'required',
                'precio_actual' => 'required|numeric',
                'imagen_curso' => 'nullable|image|mimes:jpeg,png,jpg,svg,webp|max:4000',
                'cantidad_cursos' => 'nullable|integer',
                'cantidad_paquetes' => 'nullable|integer',
                'descripcion' => 'required',
                // 'corona' => 'required|boolean', # Validar input 'corona' (boolean)
                'contenido' => 'required',
            ], $messages);

            $nombre = $request->input('nombre_curso');
            $precio_actual = $request->input('precio_actual');
            $precio_original = $request->input('precio_original');

            # Verificar si se envio una imagen
            $filename = "";
            if (!$request->imagen_curso == null) {
                $filename = uniqid() . '_' . time() . '.' . $request->imagen_curso->getClientOriginalExtension();
                $request->imagen_curso->move(public_path('images'), $filename);
            }

            $modulo = $request->input('cantidad_cursos');
            $packs = $request->input('cantidad_paquetes');
            $descripcion = $request->input('descripcion');
            $corona = $request->filled('corona'); #boolean
            $contenido = $request->input('contenido');

            # Obtener el maximo de la posicion

            $data = array(
                'Nombre' => $nombre,
                "Precio" => $precio_actual,
                "Precio_sin_descuento" => $precio_original,
                "Modulo" => $modulo, # Cantidad de cursos
                "CantidadPaquetes" => $packs, # Cantida de paquetes
                "Contenido" => $contenido,
                "Descripcion" => $descripcion,
                "Instructor_idInstructor" => 1,
                "Categoria_idCategoria" => 1,
                "numerocurso" => !$corona ? 1 : 0,
                "Estado" => 1,
                "tipoCurso" => null,
            );

            if (!$filename == "") {
                $data["Imagen"] = "images/" . $filename;
            }

            DB::table('Curso')->where('idCurso', $request->id_curso)->update($data);
            DB::commit();
            return redirect()->back()->with('success', 'Se actualizó el curso');
        } catch (ValidationException $e) {
            DB::rollback();
            // Manejo del error de validación
            return back()->withInput()->withErrors($e->errors());
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al actualizar el curso: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al actualizar el curso. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function desactivarCurso($id_curso)
    {
        try {
            DB::beginTransaction();
            DB::table('Curso')->where('idCurso', $id_curso)->update(['Estado' => 2]);

            DB::commit();
            return redirect()->back()->with('success', 'Se eliminó el curso');
        } catch (\Exception $e) {
            DB::rollback();

            return back()->withInput()->withErrors([
                'error_crear' => 'Error al eliminar el curso. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function cambiarOrderCurso($id_curso, $direccion)
    {
        DB::beginTransaction();
        try {
            # Obtener posicion del curso actual
            $posicion = DB::table('Curso')->where('idCurso', $id_curso)->value('posicion');

            if ($direccion == 1) { # Mostrar curso un puesto antes
                $nueva_posicion = DB::table('Curso')
                    ->whereRaw('CAST(posicion AS UNSIGNED) < ?', [$posicion])
                    ->max(DB::raw('CAST(posicion AS UNSIGNED)'));
            } else { # Mostrar curso un puesto despues
                $nueva_posicion = DB::table('Curso')
                    ->whereRaw('CAST(posicion AS UNSIGNED) > ?', [$posicion])
                    ->min(DB::raw('CAST(posicion AS UNSIGNED)'));
            }
            $id_curso_cambio = DB::table('Curso')->where('posicion', $nueva_posicion)->value('idCurso');
            if ($id_curso_cambio == null) { # Verificar si existe un curso con la posicion a cambiar
                return response()->json(['error' => 'No se puede cambiar el orden del curso']);
            }

            # Actualizar posicion del curso actual
            DB::table('Curso')->where('idCurso', $id_curso)->update(['posicion' => $nueva_posicion]);
            # Actualizar posicion del curso anterior
            DB::table('Curso')->where('idCurso', $id_curso_cambio)->update(['posicion' => $posicion]);

            DB::commit();
            return redirect()->back()->with('success', 'Se cambio el orden del curso');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al cambiar el orden del curso: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al cambiar el orden del curso. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function obtenerSecciones($id_curso)
    {
        return DB::table('Seccion')
            ->join('Curso_has_Seccion', 'Seccion.idSeccion', '=', 'Curso_has_Seccion.Seccion_idSeccion')
            ->select(
                'Seccion.Nombre as nombre',
                'Seccion.titulo',
                'Seccion.subtitulo',
                'Seccion.idSeccion as id'
            )
            ->where('Curso_has_Seccion.Curso_idCurso', $id_curso)
            ->paginate(5);
    }

    public function registrarSeccion(Request $request)
    {
        session()->put('redirigido_step2', true);
        try {
            DB::beginTransaction();
            $messages = [
                'nombre_seccion.required' => 'El nombre de la seccion es requerido.',
                'archivo_seccion.required' => 'El archivo de la seccion es requerido.',
                'archivo_seccion.mimes' => 'El archivo debe ser en formato JPEG, PNG, JPG, SVG o WEBP.',
                'archivo_seccion.max' => 'El archivo excede el tamaño permitido.',
                'titulo_seccion.required' => 'El titulo de la seccion es requerido.',
                'subtitulo_seccion.required' => 'El subtitulo de la seccion es requerido.',
                'id_curso.required' => 'Error al registrar la seccion. Por favor, inténtalo de nuevo. (001)',
                'id_curso.integer' => 'Error al registrar la seccion. Por favor, inténtalo de nuevo. (002)'
            ];
            $request->validate([
                'nombre_seccion' => 'required',
                'archivo_seccion' => 'required|mimes:jpeg,png,jpg,svg,webp,gif,mp4,mov,ogg,qt|max:20000',
                'titulo_seccion' => 'required',
                'subtitulo_seccion' => 'required',
                'id_curso' => 'required|integer',
            ], $messages);

            $nombre = $request->input('nombre_seccion');
            $imagen_url = "";

            if (!$request->archivo_seccion == null) {
                $filename = uniqid() . '_' . time() . '.' . $request->archivo_seccion->getClientOriginalExtension();
                $request->archivo_seccion->move(public_path('images'), $filename);
                $imagen_url = "images/" . $filename;
            }
            $titulo = $request->input('titulo_seccion');
            $subtitulo = $request->input('subtitulo_seccion');
            $id_curso = $request->input('id_curso');

            $data = array(
                'Nombre' => $nombre,
                'Estado' => 1, # Estado de la seccion
                "Imagen" => $imagen_url,
                'titulo' => $titulo,
                'subtitulo' => $subtitulo,
            );

            $id_seccion = DB::table('Seccion')->insertGetId($data);
            $data = array(
                'Curso_idCurso' => $id_curso,
                'Seccion_idSeccion' => $id_seccion,
            );

            DB::table('Curso_has_Seccion')->insert($data);
            DB::commit();

            # avanzar al siguiente step
            session()->put('registrarSeccion', true);
            DB::commit();
            return redirect()->back()->with('success', 'Se registró la seccion');
        } catch (ValidationException $e) {
            DB::rollback();
            // Manejo del error de validación
            return back()->withInput()->withErrors($e->errors());
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al registrar la seccion: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al registrar la seccion. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function eliminarSeccion($id_seccion)
    {
        session()->put('redirigido_step2', true);
        DB::beginTransaction();
        try {
            DB::table('Seccion_has_Clase')->where('Seccion_idSeccion', $id_seccion)->delete();
            DB::table('Clase')->whereNotIn('idClase', function ($query) {
                $query->select('Clase_idClase')->from('Seccion_has_Clase');
            })->delete();

            DB::table('Curso_has_Seccion')->where('Seccion_idSeccion', $id_seccion)->delete();
            DB::table('Seccion')->where('idSeccion', $id_seccion)->delete();

            DB::commit();
            return redirect()->back()->with('success', 'Se eliminó la seccion');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al eliminar la seccion: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al eliminar la seccion. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function obtenerClases($id_seccion)
    {
        return DB::table('Clase')
            ->join('Seccion_has_Clase', 'Clase.idClase', '=', 'Seccion_has_Clase.Clase_idClase')
            ->select(
                'Clase.Nombre as nombre',
                'Clase.Descripcion as detalle',
                'Clase.idClase as id',
                'Clase.Multimedia as url_clase',
                'Clase.otro as url_material',
            )
            ->where('Seccion_has_Clase.Seccion_idSeccion', $id_seccion)
            ->paginate(5);
    }

    public function pasoCrearClase(Request $request)
    {
        $messages = [
            'id_curso.required' => 'Acceso denegado.',
            'id_curso.integer' => 'Acceso denegado.',
            'id_seccion.required' => 'Acceso denegado.',
            'id_seccion.integer' => 'Acceso denegado.',
        ];

        $request->validate([
            'id_curso' => 'required|integer',
            'id_seccion' => 'required|integer',
        ], $messages);

        session()->put('redirigido_step3', true);
        session()->put('tab', 'step3');
        session()->put('id_seccion', $request->id_seccion);

        return redirect()->route('creacion-cursos');
    }

    public function pasoCrearSeccion(Request $request)
    {
        $messages = [
            'id_curso.required' => 'Acceso denegado.',
            'id_curso.integer' => 'Acceso denegado.',
        ];

        $request->validate([
            'id_curso' => 'required|integer',
        ], $messages);

        session()->put('redirigido_step2', true);
        session()->put('tab', 'step2');
        session()->put('id_curso', $request->id_curso);

        return redirect()->route('creacion-cursos');
    }

    public function registrarClase(Request $request)
    {
        session()->put('redirigido_step3', true);
        DB::beginTransaction();
        try {
            $messages = [
                'nombre_clase.required' => 'El nombre de la clase es requerido.',
                'detalle_clase.required' => 'El detalle de la clase es requerido.',
                'url_clase.required_without' => 'La URL de la clase es requerida
            si el material de descarga no está presente.',
                'url_material.required_without' => 'El material de la clase es requerido
            si la URL de la clase no está presente.',
                'id_seccion.required' => 'Error al registrar la clase. Por favor, inténtalo de nuevo. (001)',
                'id_seccion.integer' => 'Error al registrar la clase. Por favor, inténtalo de nuevo. (002)'
            ];

            $request->validate([
                'nombre_clase' => 'required',
                'detalle_clase' => 'required',
                'url_clase' => 'required_without:url_material',
                'url_material' => 'required_without:url_clase',
                'id_seccion' => 'required|integer',
            ], $messages);

            $nombre = $request->input('nombre_clase');
            $detalle = $request->input('detalle_clase');
            $multimedia = $request->input('url_clase');
            $otro = $request->input('url_material');

            $data = array(
                'Nombre' => $nombre,
                'Descripcion' => $detalle,
                'Estado' => 1,
                "Multimedia" => $multimedia,
                'otro' => $otro ?? "",
            );

            $id_clase = DB::table('Clase')->insertGetId($data);

            $id_seccion = $request->input('id_seccion');
            $data = array(
                'Seccion_idSeccion' => $id_seccion,
                'Clase_idClase' => $id_clase,
            );

            DB::table('Seccion_has_Clase')->insert($data);
            DB::commit();
            return redirect()->back()->with('success', 'Se registró la clase');
        } catch (ValidationException $e) {
            DB::rollback();
            return back()->withInput()->withErrors($e->errors());
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al registrar la clase: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al registrar la clase. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function eliminarClase($id_clase)
    {
        session()->put('redirigido_step3', true);
        DB::beginTransaction();
        try {
            DB::table('Seccion_has_Clase')->where('Clase_idClase', $id_clase)->delete();
            DB::table('Clase')->where('idClase', $id_clase)->delete();
            DB::commit();
            return redirect()->back()->with('success', 'Se eliminó la clase');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error al eliminar la clase: ' . $e->getMessage());
            return back()->withInput()->withErrors([
                'error_crear' => 'Error al eliminar la clase. Por favor, inténtalo de nuevo.'
            ]);
        }
    }

    public function obtenerTextoMesEs($mes)
    {
        $mes_es = "";
        switch ($mes) {
            case 'January':
                $mes_es = 'Enero';
                break;
            case 'February':
                $mes_es = 'Febrero';
                break;
            case 'March':
                $mes_es = 'Marzo';
                break;
            case 'April':
                $mes_es = 'Abril';
                break;
            case 'May':
                $mes_es = 'Mayo';
                break;
            case 'June':
                $mes_es = 'Junio';
                break;
            case 'July':
                $mes_es = 'Julio';
                break;
            case 'August':
                $mes_es = 'Agosto';
                break;
            case 'September':
                $mes_es = 'Septiembre';
                break;
            case 'October':
                $mes_es = 'Octubre';
                break;
            case 'November':
                $mes_es = 'Noviembre';
                break;
            case 'December':
                $mes_es = 'Diciembre';
                break;
            default:
                $mes_es = '';
                break;
        }
        return $mes_es;
    }

    public function generarCertificadoPDF(Request $request)
    {
        try {
            $messages = [
                'nombre_graduado.required' => 'El nombre del graduado es requerido.',
                'email_graduado.required' => 'El email del graduado es requerido.',
                'email_graduado.email' => 'El email del graduado debe ser un email válido.',
                'fecha_certificacion.required' => 'La fecha de certificación es requerida.',
                'fecha_certificacion.date' => 'La fecha de certificación debe ser una fecha válida.',
                'id_curso_graduacion.required' => 'El curso de graduación es requerido.',
            ];

            $request->validate([
                'nombre_graduado' => 'required',
                'email_graduado' => 'required|email',
                'fecha_certificacion' => 'required|date_format:d/m/Y',
                'id_curso_graduacion' => 'required|integer',
            ], $messages);

            # ucwords: Convertir a mayuscula la primera letra de cada palabra
            $nombre_graduado = mb_convert_case($request->input('nombre_graduado'), MB_CASE_TITLE);
            $email_graduado = $request->input('email_graduado');
            $fecha_certificacion = $request->input('fecha_certificacion');
            $id_curso_graduacion = $request->input('id_curso_graduacion');

            $curso_nombre = DB::table('Curso')->where('idCurso', $id_curso_graduacion)->value('Nombre');

            $fechaObj = DateTime::createFromFormat('d/m/Y', $fecha_certificacion); # crear objeto fecha
            $new_date = date("m/d/Y", $fechaObj->getTimestamp()); # convertir fecha a formato americano
            $mes = $this->obtenerTextoMesEs(date("F", strtotime($new_date)));

            # obtener fecha actual
            $fecha_actual = date("d") . ' de ' . $this->obtenerTextoMesEs(date("F")) . ' del ' . date("Y");

            $data_pdf = [
                'nombre_graduado' => $nombre_graduado,
                'fecha_certificacion' => $fecha_certificacion,
                'curso_graduacion' => $curso_nombre,
                'mes_graduacion' => $mes,
                'anio_graduacion' => date("Y", $fechaObj->getTimestamp()),
                'fecha_actual' => $fecha_actual,
            ];

            $pdf = PDF::loadView('cursos.certificado', $data_pdf)->setPaper('A4', 'landscape');
            $pdfOutput = $pdf->output();

            $subject = "Ilumina Arquitectura - Certificado";
            Mail::send(
                'cursos/email_certificado',
                [
                    'nombre_graduado' => $nombre_graduado,
                    'curso_graduacion' => $curso_nombre,
                ],
                function ($msj) use ($subject, $email_graduado, $pdfOutput) {
                    $msj->from("soporte@iluminaarquitectura.com", "ilumina arquitectura");
                    $msj->subject($subject);
                    $msj->to($email_graduado);
                    $msj->attachData($pdfOutput, 'certificado.pdf', [
                        'mime' => 'application/pdf',
                    ]);
                }
            );

            // return $pdf->stream(); # Stream PDF
            return redirect()->back()->with('success', 'Se envió el certificado');
        } catch (ValidationException $e) {
            DB::rollback();
            // Manejo del error de validación
            return back()->withInput()->withErrors($e->errors());
        } catch (\Exception $e) {
            Log::error('[Error al enviar el certificado] ' . $e->getMessage() . '. ' . $e->getFile() . ':' . $e->getLine());
            return back()->withInput()->withErrors([
                'error_enviar' => 'Hubo un error al enviar el certificado.'
            ]);
        }
    }

    public function listadoCursos(Request $request)
    {
        $paginaActual = $request->get('page', 1);
        $elementosPorPagina = 10; // Cambia esto por el número de elementos por página que estás usando
        $cursos = $this->obtenerCursos($elementosPorPagina);
        return view('admin/listado_cursos', compact('cursos'), compact('paginaActual', 'elementosPorPagina'));
    }

    public function creacionCertificados()
    {
        # obtener lista de cursos
        $cursos = DB::table('Curso')
            ->select(
                'Curso.Nombre as nombre',
                'Curso.idCurso as id'
            )
            ->whereIn('Curso.Estado', [0, 1])
            ->orderByRaw('nombre')
            ->get();
        return view('admin/creacion_certificados', compact('cursos'));
    }

    public function actualizacionPantallaPrincipal()
    {
        return view('admin/actualizacion_pantalla_principal');
    }
}
